import argparse
import logging

from pgtoolkit import pgmanager, config
from pgtoolkit.coloredformatter import ColoredFormatter
from pgtoolkit.highlight import highlight


class ToolBase:
    def __init__(self, name, desc):
        self.parser = argparse.ArgumentParser(description=desc)
        self.parser.add_argument('-d', dest='debug', action='store_true',
            help='Debug mode - print database queries.')
        
        self.config = config.ConfigParser()
        self.config.add_argument('databases', type=dict)
        self.config.add_argument('meta_db')
        self.config.add_argument('meta_query')
        
        self.pgm = pgmanager.get_instance()
        
    def init(self):
        self.config.load('pgtoolkit.conf')
        self.args = self.parser.parse_args()
        if self.args.debug:
            handler = logging.StreamHandler()
            format = ColoredFormatter(
                highlight(1,7,0)+'%(asctime)s %(levelname)-5s'+highlight(0)+' %(message)s', '%H:%M:%S')
            handler.setFormatter(format)
            handler.setLevel(logging.DEBUG)
            logger = logging.getLogger('pgmanager')
            logger.addHandler(handler)

    def buildconn(self, name, targetname):
        with self.pgm.cursor('meta') as curs:
            curs.execute(self.config.meta_query, [targetname])
            row = curs.fetchone_dict()
            curs.connection.commit()
            
            if not row:
                raise Exception('Unknown database "%s"' % targetname)
            
            self.pgm.create_conn(name=name, **row)

    def prepareconns(self, *args):        
        if self.config.meta_db:
            self.pgm.create_conn(name='meta', dsn=self.config.meta_db)
            for name in args:
                self.buildconn(name, self.args.__dict__[name])
            self.pgm.close_conn('meta')


class SimpleTool(ToolBase):
    def __init__(self, name, desc):
        ToolBase.__init__(self, name, desc)        
        self.parser.add_argument('target', metavar='target', type=str, help='Target database')
      
    def init(self):
        ToolBase.init(self)
        self.prepareconns('target')


class SrcDstTool(ToolBase):
    def __init__(self, name, desc):
        ToolBase.__init__(self, name, desc)        
        self.parser.add_argument('src', metavar='source', type=str, help='Source database')
        self.parser.add_argument('dst', metavar='destination', type=str, help='Destination database')
      
    def init(self):
        ToolBase.init(self)
        self.prepareconns('src', 'dst')

