/*
 * container.cc: Container class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2007  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "common.h"
#include "container.h"
#include "serialize.h"

void Container::addShape(Shape* aShape)
{
	const Float e = Eps;
	if (shapes.size() == 0) {
		/* initialize bounding box */
		bbox = aShape->get_bbox();
		const Vector3 E(e, e, e);
		bbox = BBox(bbox.L - E, bbox.H + E);
	} else {
		/* adjust bounding box */
		BBox shapebb = aShape->get_bbox();
		if (shapebb.L.x - e < bbox.L.x)  bbox.L.x = shapebb.L.x - e;
		if (shapebb.L.y - e < bbox.L.y)  bbox.L.y = shapebb.L.y - e;
		if (shapebb.L.z - e < bbox.L.z)  bbox.L.z = shapebb.L.z - e;
		if (shapebb.H.x + e > bbox.H.x)  bbox.H.x = shapebb.H.x + e;
		if (shapebb.H.y + e > bbox.H.y)  bbox.H.y = shapebb.H.y + e;
		if (shapebb.H.z + e > bbox.H.z)  bbox.H.z = shapebb.H.z + e;
	}
	shapes.push_back(aShape);
};

Shape *Container::nearest_intersection(const Shape *origin_shape, const Ray &ray,
        Float &nearest_distance)
{
	Shape *nearest_shape = NULL;
	ShapeList::iterator shape;
	for (shape = shapes.begin(); shape != shapes.end(); shape++)
		if (*shape != origin_shape && (*shape)->intersect(ray, nearest_distance))
			nearest_shape = *shape;
	return nearest_shape;
}

ostream & Container::dump(ostream &st)
{
	st << "(container," << shapes.size();
	ShapeList::iterator shape;
	for (shape = shapes.begin(); shape != shapes.end(); shape++)
	{
		int idx;
		if (!shape_index.get(*shape, idx))
			st << "," << **shape;
	}
	return st << ")";
}
