/*
 * kdtree.h: KdTree class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef KDTREE_H
#define KDTREE_H

#include <iostream>
#include <fstream>
#include <assert.h>

#include "container.h"
#include "vector.h"
#include "scene.h"

using namespace std;

/**
 * a node of kd-tree
 */
class KdNode
{
	Float split;
	union {
		KdNode *children;
		ShapeList *shapes;
		int flags; /* 0,1,2 => x,y,z; 3 => leaf */
	};
public:
	KdNode() { shapes = new ShapeList(); assert((flags & 3) == 0); setLeaf(); };
	~KdNode();

	void setLeaf() { flags |= 3; };
	bool isLeaf() const { return (flags & 3) == 3; };

	void setAxis(int aAxis) { flags &= ~3; flags |= aAxis; };
	int getAxis() const { return flags & 3; };

	void setSplit(Float aSplit) { split = aSplit; };
	const Float& getSplit() const { return split; };

	void setChildren(KdNode *node) { children = node; assert((flags & 3) == 0); };
	KdNode* getLeftChild() const { return (KdNode*)((off_t)children & ~3); };
	KdNode* getRightChild() const { return (KdNode*)((off_t)children & ~3) + 1; };

	ShapeList* getShapes() const { return (ShapeList*)((off_t)shapes & ~3); };
	void addShape(Shape* aShape) { getShapes()->push_back(aShape); };
};

/**
 * kd-tree
 */
class KdTree: public Container
{
	KdNode *root;
	bool built;
	int max_depth;

	void recursive_build(KdNode *node, BBox bbox, int maxdepth);
	void recursive_load(istream &st, KdNode *node);
public:
	KdTree() : Container(), root(NULL), built(false), max_depth(32) {};
	~KdTree() { if (root) delete root; };
	void addShape(Shape* aShape) { Container::addShape(aShape); built = false; };
	Shape *nearest_intersection(const Shape *origin_shape, const Ray &ray,
		Float &nearest_distance);
	void packet_intersection(const Shape **origin_shapes, const RayPacket &rays,
		Float *nearest_distances, Shape **nearest_shapes);
	void optimize() { build(); };
	void build();
	bool isBuilt() const { return built; };
	KdNode *getRootNode() const { return root; };
	void setMaxDepth(int md) { max_depth = md; };

	ostream & dump(ostream &st);
	istream & load(istream &st, Material *mat);
};

#endif
