/**
 * @file  common.h
 * @brief Common functions and definitions
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef COMMON_H
#define COMMON_H

#include "config.h"

#include <stdio.h>
#include <math.h>
#include <float.h>
#include <pthread.h>
#include <stdint.h>
#include <string>

using namespace std;


#ifdef PYRIT_DOUBLE
# define Float double
# define Eps DBL_EPSILON
# define Inf DBL_MAX
# define PI M_PI
#else
# define Float float
# define Eps 2e-6f
# define Inf FLT_MAX
# define PI (float)M_PI
#endif

#define NORETURN __attribute__((noreturn))

/**
 * verbosity level
 *
 * - 0: only errors and warnings (E, W)
 * - 1: major status messages (*)
 * - 2: minor status, progress (-)
 * - 3: debug messages (D)
 * - 4: thread debug
 *
 * default is 2
 */
extern int pyrit_verbosity;

/**
 * print status or debugging message
 * @param[in] vlevel    verbosity level of the message
 * @see pyrit_verbosity
 */
void dbgmsg(const int vlevel, const char *format, ...);

/** template for minimum of three numbers */
template<typename Type> const Type &min3(const Type &a, const Type &b, const Type &c)
{
	if (a <= b)
	{
		if (a <= c)
			return a;
		else
			return c;
	}
	else
	{
		if (b <= c)
			return b;
		else
			return c;
	}
}

/** template for maximum of three numbers */
template<typename Type> const Type &max3(const Type &a, const Type &b, const Type &c)
{
	if (a >= b)
	{
		if (a >= c)
			return a;
		else
			return c;
	}
	else
	{
		if (b >= c)
			return b;
		else
			return c;
	}
}

/** trim whitespace from beginning and end of string */
inline void trim(string& s)
{
	const char *ws = " \n";
	string::size_type pos = s.find_last_not_of(ws);
	if (pos != string::npos)
	{
		s.erase(pos + 1);
		pos = s.find_first_not_of(ws);
		if (pos != string::npos)
			s.erase(0, pos);
	}
	else
		s.erase(s.begin(), s.end());
}

#endif
