/**
 * @file  container.h
 * @brief Spatial container for shapes
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef CONTAINER_H
#define CONTAINER_H

#include <vector>

#include "shapes.h"

using namespace std;

/**
 * General container for shapes.
 *
 * Does very simple intersection test:
 * all shapes are tested and the nearest intersection is returned.
 */
class Container
{
protected:
	BBox bbox;
public:
	ShapeList shapes;

	Container(): bbox(), shapes() {};
	virtual ~Container() {};

	/** add pointer to shape to the container */
	virtual void addShape(const Shape* aShape);
	//void addShapeNoExtend(const Shape* aShape) { shapes.push_back(aShape); };

	/**
	 * find nearest intersection with shapes in container
	 * @param[in] origin_shape  this shape should be avoided from the test
	 * @param[in] ray           the ray
	 * @param[out] nearest_distance  the nearest allowd distance of intersection;
	 *                               it is updated when closer intersection is found
	 * @return intersected shape or NULL if no intersection was found
	 */
	virtual const Shape *nearest_intersection(const Shape *origin_shape, const Ray &ray,
		Float &nearest_distance);

	/** intersect with whole ray packet */
#ifndef NO_SIMD
	virtual void packet_intersection(const Shape* const* origin_shapes, const RayPacket &rays,
		Float *nearest_distances, const Shape** nearest_shapes);
#endif

	/** build acceleration structures */
	virtual void optimize() {};

	/** get reference to the shape list */
	ShapeList & getShapes() { return shapes; };

	/** write textual representation of the acceleration structure
	 * and shapes in list to the stream */
	virtual ostream & dump(ostream &st) const;
};

#endif
