/**
 * @file  sampler.h
 * @brief Screen sample generation and image reconstruction
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SAMPLER_H
#define SAMPLER_H

#include "common.h"
#include "vector.h"
#include "pixmap.h"

using namespace std;

/**
 * A structure containing sample's coordinates.
 */
class Sample
{
public:
	//@{
	/**
	 * Generated coordinates.
	 *
	 * These are coordinates for use by Camera.
	 * Middle of the screen plane has [0,0],
	 * top of the screen has y=-0.5, left pixels has x<0
	 * and the rest of samples are aligned to regular grid,
	 * so the vertical size is always one and horizontal size
	 * can be wider depending on aspect ratio.
	 */
	Float x,y;
	//@}

	//@{
	/**
	 * Auxiliary coordinates used internally by sampler.
	 *
	 * Screen coordinates sx, sy are integer coordinates of screen pixels,
	 * osa_samp is used by oversampling to remember current sample.
	 */
	int sx,sy,osa_samp;
	//@}
};

/**
 * A abstract sampler.
 *
 * It generates screen samples in coordinates between [-0.5..0.5] for height
 * and [-w/h/2..w/h/2] for width. It works in phases: initSampleSet returns
 * number of samples for each phase, then samples can be generated using
 * nextSample method. The resulting colour of each sample should be returned
 * via saveSample method. The sampler should save the results to given buffer
 * and decide if another phase is needed. When the picture is complete,
 * initSampleSet returns zero and picture can be read from buffer.
 */
class Sampler
{
protected:
	Pixmap pixmap;    ///< A pixmap for resulting image.
	bool packetable;  ///< True if the generated samples are usable with ray packets.
public:
	/** Constructor with user provided image buffer. */
	Sampler(Float *buffer, const int &w, const int &h):
		pixmap(buffer, w, h), packetable(false) {};

	/** With this constructor, image buffer is allocated as specified by w,h. */
	Sampler(const int &w, const int &h):
		pixmap(w, h), packetable(false) {};

	virtual ~Sampler() {};

	/** Switch to new user provided buffer. */
	void resetBuffer(Float *buffer, int &w, int &h)
		{ pixmap.setData(buffer, w, h); };

	/**
	 * Initialize the sampler.
	 *
	 * This must be called first when starting new image.
	 */
	virtual void init() = 0;

	/**
	 * Prepare for next phase.
	 *
	 * Must be called before reading samples.
	 * @retval N approximate number of samples to be generated in the phase
	 * @retval 0 last phase, image is complete
	 */
	virtual int initSampleSet() = 0;

	/**
	 * Generate next sample.
	 *
	 * @param[out] s  The sample.
	 * @retval false  No more samples in this phase.
	 */
	virtual bool nextSample(Sample *s) = 0;

	/**
	 * Save the color of the sample.
	 *
	 * @param[in] samp  A sample.
	 * @param[in] col   Color resulted from ray tracing the sample.
	 */
	virtual void saveSample(const Sample &samp, const Colour &col) = 0;

	/** True if generated samples are usable with ray packets. */
	bool packetableSamples() { return packetable; };

	/** Get the pixmap. */
	const Pixmap &getPixmap() const { return pixmap; };
};

/**
 * Default sampler.
 *
 * Implements basic adaptive subsampling and oversampling.
 */
class DefaultSampler: public Sampler
{
	int phase;      ///< Current phase
	int subsample;  ///< Subsampling mode
	int oversample; ///< Oversampling mode
	//@{
	/** Properties of last generated sample. */
	int sx,sy,osa_samp;
	//@}
public:
	/** Constructor with user provided image buffer. */
	DefaultSampler(Float *buffer, const int &w, const int &h):
		Sampler(buffer, w, h), phase(-1), subsample(0), oversample(0) {};

	/** With this constructor, image buffer is allocated as specified by w,h. */
	DefaultSampler(const int &w, const int &h):
		Sampler(w, h), phase(-1), subsample(0), oversample(0) {};

	/* Implement virtuals from Sampler. */
	void init();
	int initSampleSet();
	bool nextSample(Sample *s);
	void saveSample(const Sample &samp, const Colour &col);

	/**
	  * Set subsampling mode.
	  *
	  * @param[in] sub  0,1 = no, 1+ = size of sampling grid
	  */
	void setSubsample(int sub) { subsample = sub; };
	/** Get subsampling mode. @see setSubsample */
	int getSubsample() { return subsample; };

	/**
	  * Set oversampling mode.
	  *
	  * @param[in] osa  0 = no, 1 = 4x, 2 = 9x, 3 = 16x
	  */
	void setOversample(int osa) { oversample = osa; };
	/** Get oversampling mode. @see setOversample */
	int getOversample() { return oversample; };
};

#endif
