/**
 * @file  octree.h
 * @brief Octree class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef OCTREE_H
#define OCTREE_H

#include "container.h"
#include "vector.h"
#include "scene.h"

#include <assert.h>

using namespace std;

/**
 * a node of octree
 */
class OctreeNode
{
    union {
		OctreeNode *children; // pointer to first of eight children
		ShapeList *shapes;    // pointer to shape array, if this is leaf
		off_t leaf;             // leaf indicator (bit 0)
	};
public:
	OctreeNode()
	{
		shapes = new ShapeList();
		assert(sizeof(off_t)==sizeof(void*) && !isLeaf());
		setLeaf();
	};
	~OctreeNode();

	bool isLeaf() { return leaf & 1; };
	void setLeaf() { leaf = leaf | 1; };

	void makeChildren() { children = new OctreeNode[8]; assert(!isLeaf()); }; // this also cleans leaf bit
	OctreeNode *getChild(const int num) { assert(!isLeaf()); return children + num; };

	void addShape(Shape* aShape) { getShapes()->push_back(aShape); };
	ShapeList *getShapes() { return (ShapeList*)((size_t)shapes & ~(size_t)1); };
	void setShapes(ShapeList *const ashapes) { shapes = ashapes; assert(!isLeaf()); setLeaf(); };

	void subdivide(const BBox &bbox, int maxdepth);
};

/**
 * optimized octree
 */
class Octree: public Container
{
	OctreeNode *root;
	const int max_depth;
	bool built;
public:
	Octree() : Container(), root(NULL), max_depth(10), built(false) {};
	Octree(int maxdepth) : Container(), root(NULL), max_depth(maxdepth), built(false) {};
	~Octree() { if (root) delete root; };
	void addShape(Shape* aShape) { Container::addShape(aShape); built = false; };
	Shape *nearest_intersection(const Shape *origin_shape, const Ray &ray,
		Float &nearest_distance);
	void optimize() { build(); };
	void build();
	void save(ostream &str, OctreeNode *node = NULL) {};
	void load(istream &str, OctreeNode *node = NULL) {};
};

#endif
