/**
 * @file  mempool.h
 * @brief Memory pool for aligned allocation
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MEMPOOL_H
#define MEMPOOL_H

#include "common.h"

#include <string.h>

/**
 * Memory pool template
 */
template <typename Type>
class MemoryPool
{
	void *mem;
	size_t cur, size, typesize, align;

	void init()
	{
		typesize = ((sizeof(Type)-1)/align+1)*align;
#ifndef NO_SIMD
		mem = (Type *)_mm_malloc(size * typesize, align);
#else
		mem = (Type *)malloc(size * typesize);
#endif
	};
public:
	/**
	 * Construct memory pool with default alignment (16)
	 * @see MemoryPool(size_t, size_t)
	 */
	MemoryPool(const size_t inisize):
		cur(0), size(inisize), align(16) { init(); };

	/**
	 * Construct memory pool
	 * @param[in] inisize  Initial capacity in Type count
	 * @param[in] inialign Set memory alignement of items
	 */
	MemoryPool(const size_t inisize, const size_t inialign):
		cur(0), size(inisize), align(inialign) { init(); };
	~MemoryPool()
	{
#ifndef NO_SIMD
		_mm_free(mem);
#else
		free(mem);
#endif
	};

	/**
	 * Allocate memory for one item of type Type
	 * @return Pointer to allocated memory
	 */
	void *alloc()
	{
		if (cur == size)
		{
#ifndef NO_SIMD
			void *newmem = _mm_malloc(2 * size * typesize, align);
#else
			void *newmem = malloc(2 * size * typesize);
#endif
			memcpy(newmem, mem, size * typesize);
			mem = newmem;
			size *= 2;
		}
		return (void*)((size_t)mem + typesize*(cur++));
	};
};

#endif
