/*
 * scene.cc: screen sample generation and image reconstruction
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include <math.h>

#include "common.h"
#include "scene.h"

void DefaultSampler::init()
{
	phase = 0;
}

int DefaultSampler::initSampleSet()
{
	static const int gridsamples[] = {1,5,9,16};
	const int samples = gridsamples[oversample];
	if ( phase == 0 )
	{
		cout << "phase 1" << endl;
		phase++;
		return w*h*samples;
	}
	else if ( phase == 1 && oversample )
	{
		cout << "phase 2" << endl;
		phase = -1;
		Float *buf;
		for (buf = buffer; buf != buffer + w*h*3; buf++)
			*buf = *buf * (1.0/samples);
		return 0;
	}
	else
	{
		phase = -1;
		return 0;
	}
}

Sample* DefaultSampler::nextSample(Sample *prev)
{
	DefaultSample *s = new DefaultSample;

	/* grid oversampling */
	static const int gridsamples[] = {1,5,9,16};
	static const Float osa5x[] = {0.0, -0.4, +0.4, +0.4, -0.4};
	static const Float osa5y[] = {0.0, -0.4, -0.4, +0.4, +0.4};
	static const Float osa9x[] = {-0.34,  0.00, +0.34,
		-0.34,  0.00, +0.34, -0.34,  0.00, +0.34};
	static const Float osa9y[] = {-0.34, -0.34, -0.34,
			0.00,  0.00,  0.00, +0.34, +0.34, +0.34};
	static const Float osa16x[] = {-0.375, -0.125, +0.125, +0.375,
		-0.375, -0.125, +0.125, +0.375, -0.375, -0.125, +0.125, +0.375,
		-0.375, -0.125, +0.125, +0.375};
	static const Float osa16y[] = {-0.375, -0.375, -0.375, -0.375,
		-0.125, -0.125, -0.125, -0.125, +0.125, +0.125, +0.125, +0.125,
		+0.375, +0.375, +0.375, +0.375};
	static const Float *osaSx[] = {NULL, osa5x, osa9x, osa16x};
	static const Float *osaSy[] = {NULL, osa5y, osa9y, osa16y};
	const int samples = gridsamples[oversample];
	const Float *osax = osaSx[oversample];
	const Float *osay = osaSy[oversample];

	if (!prev)
	{
		// first sample
		s->x = -(Float)w/h/2.0;
		s->y = -0.5;
		s->sx = 0;
		s->sy = 0;
		s->osa_samp = 0;
	}
	else
	{
		DefaultSample *sp = static_cast<DefaultSample*>(prev);

		s->osa_samp = sp->osa_samp + 1;

		if (oversample && oversample <= 3 && s->osa_samp < samples)
		{
			s->sx = sp->sx;
			s->sy = sp->sy;
			s->x = osax[s->osa_samp]/h + (Float)s->sx/h - (Float)w/h/2.0;
			s->y = osay[s->osa_samp]/h + (Float)s->sy/h - 0.5;
		}
		else
		{
			s->sx = sp->sx + 1;
			s->sy = sp->sy;
			if (s->sx >= w)
			{
				s->sx = 0;
				s->sy++;
			}
			if (s->sy >= h)
			{
				delete s;
				return NULL;
			}
			s->x = (Float)s->sx/h - (Float)w/h/2.0;
			s->y = (Float)s->sy/h - 0.5;
			s->osa_samp = 0;
		}
	}

	if (s->osa_samp == 0 && oversample && oversample <= 3)
	{
		s->x += osax[0]/h;
		s->y += osay[0]/h;
		Float *buf = buffer + 3*(s->sy*w + s->sx);
		*(buf++) = 0;
		*(buf++) = 0;
		*(buf++) = 0;
	}

	return s;
}

void DefaultSampler::saveSample(Sample *samp, Colour &col)
{
	DefaultSample *sp = static_cast<DefaultSample*>(samp);
	Float *buf = buffer + 3*(sp->sy*w + sp->sx);
	if (oversample)
	{
		*(buf+0) += col.r;
		*(buf+1) += col.g;
		*(buf+2) += col.b;
	}
	else
	{
		*(buf++) = col.r;
		*(buf++) = col.g;
		*(buf++) = col.b;
	}
}

