/*
 * kdtree.h: KdTree class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef KDTREE_H
#define KDTREE_H

#include <iostream>
#include <fstream>

#include "container.h"
#include "vector.h"
#include "scene.h"

using namespace std;

class KdNode
{
	Float split;
	short axis; /* 0,1,2 => x,y,z; 3 => leaf */
public:
	union {
		KdNode *children;
		ShapeList *shapes;
	};

	KdNode() : axis(3) { shapes = new ShapeList(); };
	~KdNode();

	void setAxis(short aAxis) { axis = aAxis; };
	short getAxis() { return axis; };

	void setSplit(Float aSplit) { split = aSplit; };
	Float getSplit() { return split; };

	void setLeaf() { axis = 3; };
	bool isLeaf() { return axis == 3; };

	KdNode *getLeftChild() { return children; };
	KdNode *getRightChild() { return children+1; };

	void addShape(Shape* aShape) { shapes->push_back(aShape); };

	void subdivide(BBox bbox, int maxdepth);
};

class KdTree: public Container
{
	KdNode *root;
	bool built;
	int max_depth;
public:
	KdTree() : Container(), root(NULL), built(false), max_depth(32) {};
	~KdTree() { if (root) delete root; };
	void addShape(Shape* aShape) { Container::addShape(aShape); built = false; };
	Shape *nearest_intersection(const Shape *origin_shape, const Ray &ray,
		Float &nearest_distance);
	void optimize() { build(); };
	void build();
	void save(ostream &str, KdNode *node = NULL);
	void load(istream &str, KdNode *node = NULL);
	void setMaxDepth(int md) { max_depth = md; };
};

#endif
