/*
 * quaternion.h: Quaternion class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef QUATERNION_H
#define QUATER_H

/**
 * helper quaternion class
 */
class Quaternion
{
public:
	union {
		struct {
			Float a,b,c,d;
		};
		struct {
			Float t,x,y,z;
		};
	};
	Quaternion(): a(0), b(0), c(0), d(0) {};
	Quaternion(const Float aa, const Float ab, const Float ac, const Float ad):
		a(aa), b(ab), c(ac), d(ad) {};
	Quaternion(const Vector& v): a(0), b(v.x), c(v.y), d(v.z) {};

	Vector toVector() { return Vector(b, c, d); };

	Quaternion normalize()
	{
		Float f = (Float)1.0f / sqrtf(a * a + b * b + c * c + d * d);
		a *= f;
		b *= f;
		c *= f;
		d *= f;
		return *this;
	};
	Float mag2() const
	{
		return (a*a + b*b + c*c + d*d);
	};
	Float mag() const
	{
		return sqrtf(mag());
	};
	friend Quaternion operator*(const Quaternion &q1, const Quaternion &q2)
	{
		return Quaternion(
			q1.a*q2.a - q1.b*q2.b - q1.c*q2.c - q1.d*q2.d,
			q1.a*q2.b + q1.b*q2.a + q1.c*q2.d - q1.d*q2.c,
			q1.a*q2.c - q1.b*q2.d + q1.c*q2.a + q1.d*q2.b,
			q1.a*q2.d + q1.b*q2.c - q1.c*q2.b + q1.d*q2.a);
	};
	friend Float dot(const Quaternion &q1, const Quaternion &q2)
	{
		return q1.a*q2.a + q1.b*q2.b + q1.c*q2.c + q1.d*q2.d;
	};
	friend Quaternion conjugate(const Quaternion &q)
	{
		return Quaternion(q.a, -q.b, -q.c, -q.d);
	};
};

#endif
