/*
 * sampler.h: screen sample generation and image reconstruction
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SAMPLER_H
#define SAMPLER_H

#include "common.h"
#include "vector.h"
#include "pixmap.h"

using namespace std;

/**
 * sample
 */
class Sample
{
	friend class Sampler;
public:
	Float x,y;
	int sx,sy,osa_samp;
};

/**
 * A abstract sampler.
 * It generates screen samples in coordinates between [-1..1] for height
 * and [-w/h..w/h] for width. It works in phases: initSampleSet returns
 * number of samples for each phase, then samples can be generated using
 * nextSample method. The resulting colour of each sample should be returned
 * via saveSample method. The sampler should save the results to given buffer
 * and decide if other phase is needed. When the picture is complete,
 * initSampleSet returns zero and picture can be read from buffer.
 */
class Sampler
{
protected:

	Pixmap pixmap;
	bool packetable;
public:
	Sampler(Float *buffer, const int &w, const int &h):
		pixmap(buffer, w, h), packetable(false) {};
	Sampler(const int &w, const int &h):
		pixmap(w, h), packetable(false) {};
	virtual ~Sampler() {};
	void resetBuffer(Float *buffer, int &w, int &h)
		{ pixmap.setData(buffer, w, h); };
	virtual void init() = 0;
	virtual int initSampleSet() = 0;
	virtual bool nextSample(Sample *s) = 0;
	virtual void saveSample(Sample &samp, Colour &col) = 0;
	bool packetableSamples() { return packetable; };
	const Pixmap &getPixmap() const { return pixmap; };
};

/**
 * Default sampler.
 * Implements basic adaptive subsampling and oversampling.
 */
class DefaultSampler: public Sampler
{
	int phase;
	int subsample;  // 0,1 = no, 1+ = size of sampling grid
	int oversample; // 0 = no, 1 = 4x, 2 = 9x, 3 = 16x
	int sx,sy,osa_samp; // current sample properties
public:
	DefaultSampler(Float *buffer, const int &w, const int &h):
		Sampler(buffer, w, h), phase(-1), subsample(0), oversample(0) {};
	DefaultSampler(const int &w, const int &h):
		Sampler(w, h), phase(-1), subsample(0), oversample(0) {};
	void init();
	int initSampleSet();
	bool nextSample(Sample *s);
	void saveSample(Sample &samp, Colour &col);

	void setSubsample(int sub) { subsample = sub; };
	int getSubsample() { return subsample; };
	void setOversample(int osa) { oversample = osa; };
	int getOversample() { return oversample; };
};

#endif
