/*
 * scene.h: classes for objects in scene
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SCENE_H
#define SCENE_H

#include <vector>
#include <typeinfo>

#include "common.h"
#include "sampler.h"
#include "vector.h"
#include "quaternion.h"

/**
 * ray
 */
class Ray
{
public:
	Vector3 o, dir;
	Ray(): o(), dir() {};
	Ray(const Vector3 &ao, const Vector3 &adir):
		o(ao), dir(adir) {};
};

/**
 * a camera
 */
class Camera
{
public:
	Vector3 eye, p, u, v;
	Float F;

	Camera(): eye(0,0,10), p(0,0,-1), u(-1,0,0), v(0,1,0), F(2.*tan(M_PI/8.)) {};
	Camera(const Vector3 &C, const Vector3 &ap, const Vector3 &au, const Vector3 &av):
		eye(C), p(ap), u(au), v(av), F(2.*tan(M_PI/8.)) {};
	Camera(const Vector3 &from, const Vector3 &lookat, const Vector3 &up):
		eye(from), F(2.*tan(M_PI/8.))
	{
		p = lookat - from; u = cross(up, p);
		p.normalize(); u.normalize();
		v = cross(p, u);
	};
	void setEye(const Vector3 &aeye) { eye = aeye; };
	void setAngle(const Float angle) { F = 2.*tan(angle/2.); };
	void rotate(const Quaternion &q);
	void move(const Float fw, const Float left, const Float up);

	Ray makeRay(Sample &samp)
	{
		Vector3 dir = p - (u*samp.x + v*samp.y)*F;
		dir.normalize();
		return Ray(eye, dir);
	};
};

/**
 * light object
 */
class Light
{
public:
	Vector3 pos;
	Colour colour;
	bool cast_shadows;

	Light():
		pos(Vector3(0,0,0)), colour(Colour(1,1,1)), cast_shadows(true) {};
	Light(const Vector3 &position, const Colour &acolour):
		pos(position), colour(acolour), cast_shadows(true) {};
	void castShadows(bool cast) { cast_shadows = cast; };
};

/**
 * axis-aligned bounding box
 */
class BBox
{
public:
	Vector3 L;
	Vector3 H;
	BBox(): L(), H() {};
	BBox(const Vector3 aL, const Vector3 aH): L(aL), H(aH) {};
	Float w() { return H.x-L.x; };
	Float h() { return H.y-L.y; };
	Float d() { return H.z-L.z; };
	bool intersect(const Ray &ray, Float &a, Float &b);
};

#endif
