/*
 * sampler.h: screen sample generation and image reconstruction
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SAMPLER_H
#define SAMPLER_H

#include "common.h"
#include "vector.h"

using namespace std;

/**
 * sample
 */
class Sample
{
public:
	Float x,y;
};

/**
 * A abstract sampler.
 * It generates screen samples in coordinates between [-1..1] for height
 * and [-w/h..w/h] for width. It works in phases: initSampleSet returns
 * number of samples for each phase, then samples can be generated using
 * nextSample method. The resulting colour of each sample should be returned
 * via saveSample method. The sampler should save the results to given buffer
 * and decide if other phase is needed. When the picture is complete,
 * initSampleSet returns zero and picture can be read from buffer.
 */
class Sampler
{
public:
	Float *buffer;
	int w,h;

	Sampler(Float *abuffer, int &aw, int &ah): buffer(abuffer), w(aw), h(ah) {};
	virtual ~Sampler() {};
	void resetBuffer(Float *abuffer, int &aw, int &ah) { buffer = abuffer; w = aw; h = ah; };
	virtual void init() = 0;
	virtual int initSampleSet() = 0;
	virtual Sample *nextSample(Sample *prev) = 0;
	virtual void saveSample(Sample *samp, Colour &col) = 0;
};

/**
 * default sample
 */
class DefaultSample: public Sample
{
	friend class DefaultSampler;
	int sx,sy;
};

/**
 * Default sampler.
 * Implements basic adaptive subsampling and oversampling.
 */
class DefaultSampler: public Sampler
{
	int phase;
	int subsample;
	int oversample; // 0 = no, 1 = 5x, 2 = 9x, 3 = 16x
public:
	DefaultSampler(Float *abuffer, int &aw, int &ah):
		Sampler(abuffer, aw, ah), phase(-1), subsample(8), oversample(0) {};
	void init();
	int initSampleSet();
	Sample *nextSample(Sample *prev);
	void saveSample(Sample *samp, Colour &col);

	void setSubsample(int sub) { subsample = sub; };
	int getSubsample() { return subsample; };
	void setOversample(int osa) { oversample = osa; };
	int getOversample() { return oversample; };
};

#endif
