/*
 * raytracer.h: Raytracer class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef RAYTRACER_H
#define RAYTRACER_H

#include <vector>
#include <queue>

#include "common.h"
#include "container.h"
#include "scene.h"

using namespace std;

class Raytracer;

struct RenderrowData {
	Raytracer *rt;
	int w;
	Vector3 eye, dfix, dx, dy;
	Float *iter;
};

/**
 * main ray tracer class
 */
class Raytracer
{
	Container *top;
	Sampler *sampler;
	Camera *camera;
	vector<Light*> lights;
	Colour bg_colour;
	int ao_samples;
	Float ao_distance, ao_angle;
	int num_threads;
	int max_depth;

	Vector3 SphereDistribute(int i, int n, Float extent, Vector3 &normal);

	Sample **sample_queue;
	int sample_queue_max, sample_queue_pos, sample_queue_count;
	bool sample_queue_end;
	pthread_mutex_t sample_queue_mutex, sampler_mutex;
	pthread_cond_t sample_queue_cond, worker_ready_cond;

	static void *raytrace_worker(void *d);
public:
	Raytracer(): top(NULL), camera(NULL), lights(), bg_colour(0.0, 0.0, 0.0),
		ao_samples(0), num_threads(2), max_depth(3)
	{
		pthread_mutex_init(&sample_queue_mutex, NULL);
		pthread_mutex_init(&sampler_mutex, NULL);
		pthread_cond_init (&sample_queue_cond, NULL);
		pthread_cond_init (&worker_ready_cond, NULL);
	};
	~Raytracer()
	{
		pthread_mutex_destroy(&sample_queue_mutex);
		pthread_mutex_destroy(&sampler_mutex);
		pthread_cond_destroy (&sample_queue_cond);
		pthread_cond_destroy (&worker_ready_cond);
	}

	void render();
	Colour raytrace(Ray &ray, int depth, Shape *origin_shape);
	void addshape(Shape *shape) { top->addShape(shape); };
	void addlight(Light *light);
	void setSampler(Sampler *sampl) { sampler = sampl; };
	void setCamera(Camera *cam) { camera = cam; };
	void setTop(Container *atop) { top = atop; };
	Container *getTop() { return top; };

	void setMaxDepth(int newdepth) { max_depth = newdepth; };

	void ambientocclusion(int samples, Float distance, Float angle);
	void setThreads(int num) { num_threads = num; };
};

#endif
