/**
 * @file  simd.h
 * @brief Abstraction of Intel SSE instruction set
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SIMD_H
#define SIMD_H

#include "common.h"

#ifndef NO_SIMD

#include <xmmintrin.h>

#define mfloat4 __m128

#define mZero  _mm_setzero_ps()
#define mOne   _mm_set_ps1(1.0f)
#define mTwo   _mm_set_ps1(2.0f)
#define mEps   _mm_set_ps1(Eps)
#define mMEps  _mm_set_ps1(-Eps)
#define mInf   _mm_set_ps1(Inf)
#define mMInf  _mm_set_ps1(-Inf)
#define mAllSet  _mm_cmplt_ps(mZero, mOne)

#define mset1 _mm_set_ps1
#define mset _mm_set_ps

#define madd _mm_add_ps
#define msub _mm_sub_ps
#define mmul _mm_mul_ps
#define mdiv _mm_div_ps
#define msqrt _mm_sqrt_ps

#define mand _mm_and_ps
#define mor  _mm_or_ps
#define mcmpgt _mm_cmpgt_ps
#define mcmplt _mm_cmplt_ps
#define mcmpge _mm_cmpge_ps
#define mcmple _mm_cmple_ps
#define mcmpeq _mm_cmpeq_ps
#define mcmpneq _mm_cmpneq_ps
#define mmin _mm_min_ps
#define mmax _mm_max_ps
#define mmovemask _mm_movemask_ps

#define mShuffle0 _MM_SHUFFLE(0,0,0,0)
#define mShuffle1 _MM_SHUFFLE(1,1,1,1)
#define mShuffle2 _MM_SHUFFLE(2,2,2,2)
#define mShuffle3 _MM_SHUFFLE(3,3,3,3)
#define mshuffle _mm_shuffle_ps

inline const mfloat4 mselect(const mfloat4& mask, const mfloat4& a, const mfloat4& b)
{
	return _mm_or_ps(_mm_and_ps(mask, a), _mm_andnot_ps(mask, b));
}

inline const mfloat4 mfastpow(const mfloat4& base, const mfloat4& exponent)
{
    __m128 denom = _mm_mul_ps(exponent, base);
    denom = _mm_sub_ps(exponent, denom);
    denom = _mm_add_ps(base, denom);
    return _mm_mul_ps(base, _mm_rcp_ps(denom));
}
#endif

#endif
