#include "raytracer.h"
#include "kdtree.h"

#include "common_sdl.h"

Camera cam(Vector(0.0f,6.0f,6.0f), Vector(0.0f,2.0f,-7.0f), Vector(0.0f,0.0f,-1.0f));
Light light(Vector(-2.0f, 10.0f, -2.0f), Colour(0.9f, 0.9f, 0.9f));

Float lx, ly, lz, cf;

void update_callback(Float*)
{
	if (lx != 0.0)
		light.pos.x += lx;
	if (ly != 0.0)
		light.pos.y += ly;
	if (lz != 0.0)
		light.pos.z += lz;
	if (cf != 0.0)
		cam.setF(cam.getF() + cf);
}

void key_callback(int key, int down)
{
	switch (key)
	{
		case SDLK_r:
			lx = -0.1f * down;
			break;
		case SDLK_t:
			lx = +0.1f * down;
			break;
		case SDLK_f:
			ly = -0.1f * down;
			break;
		case SDLK_g:
			ly = +0.1f * down;
			break;
		case SDLK_v:
			lz = -0.1f * down;
			break;
		case SDLK_b:
			lz = +0.1f * down;
			break;

		case SDLK_z:
			cf = -0.02f * down;
			break;
		case SDLK_x:
			cf = +0.02f * down;
			break;
	}
}

int main(int argc, char **argv)
{
	Raytracer rt;

	KdTree top;
	rt.setCamera(&cam);
	rt.setTop(&top);

	rt.addLight(&light);
	light.castShadows(false);

	const Colour c_white(1, 1, 1);
	const Colour c_black(0, 0, 0);

	Material mat0a(Colour(0.7f, 0.7f, 0.7f));
	mat0a.setReflectivity(0.0f);
	Box box(Vector(-12.0f, -1.2f, -20.0f), Vector(12.0f, -1.0f, 0.0f), &mat0a);
	rt.addShape(&box);

	Material mat0b(Colour(0.1f, 0.7f, 0.8f));
	mat0b.setReflectivity(0.7f);
	Box box2(Vector(-12.0f, -1.2f, -10.0f), Vector(12.0f, 10.0f, -10.2f), &mat0b);
	rt.addShape(&box2);

	Float bounds[] = {0.3f, 0.6f, 1.1f};
	Colour colours[] = {c_black, c_white, c_black};
	BoundColourMap cmap(bounds, colours);

	// spheres
	Material mat1(c_white);
	mat1.texture = new CheckersTexture(new PlanarMap(Vector(-4.5f, 2.0f, -7.0f), 0.48f), &cmap);
	Sphere sphere1(Vector(-4.5f, 2.0f, -7.0f), 1, &mat1);
	rt.addShape(&sphere1);

	Material mat2(c_white);
	mat2.texture = new CheckersTexture(new CubicMap(Vector(-1.5f, 2.0f, -7.0f), 0.48f), &cmap);
	Sphere sphere2(Vector(-1.5f, 2.0f, -7.0f), 1, &mat2);
	rt.addShape(&sphere2);

	Material mat3(c_white);
	mat3.texture = new CheckersTexture(new CylinderMap(Vector(1.5f, 2.0f, -7.0f), 0.48f), &cmap);
	Sphere sphere3(Vector(1.5f, 2.0f, -7.0f), 1, &mat3);
	rt.addShape(&sphere3);

	Material mat4(c_white);
	mat4.texture = new CheckersTexture(new SphereMap(Vector(4.5f, 2.0f, -7.0f), 0.48f), &cmap);
	Sphere sphere4(Vector(4.5f, 2.0f, -7.0f), 1, &mat4);
	rt.addShape(&sphere4);

	// cubes
	Material mat5(c_white);
	const Vector cube1_base(-4.5f, 0.0f, -7.0f);
	mat5.texture = new CheckersTexture(new PlanarMap(cube1_base, 0.48f), &cmap);
	Box cube1(cube1_base - 1.0f, cube1_base + 1.0f, &mat5);
	rt.addShape(&cube1);

	Material mat6(c_white);
	const Vector cube2_base(-1.5f, 0.0f, -7.0f);
	mat6.texture = new CheckersTexture(new CubicMap(cube2_base, 0.48f), &cmap);
	Box cube2(cube2_base - 1.0f, cube2_base + 1.0f, &mat6);
	rt.addShape(&cube2);

	Material mat7(c_white);
	const Vector cube3_base(1.5f, 0.0f, -7.0f);
	mat7.texture = new CheckersTexture(new CylinderMap(cube3_base, 0.48f), &cmap);
	Box cube3(cube3_base - 1.0f, cube3_base + 1.0f, &mat7);
	rt.addShape(&cube3);

	Material mat8(c_white);
	const Vector cube4_base(4.5f, 0.0f, -7.0f);
	mat8.texture = new CheckersTexture(new SphereMap(cube4_base, 0.48f), &cmap);
	Box cube4(cube4_base - 1.0f, cube4_base + 1.0f, &mat8);
	rt.addShape(&cube4);

	mat1.setReflectivity(0);
	mat2.setReflectivity(0);
	mat3.setReflectivity(0);
	mat4.setReflectivity(0);
	mat5.setReflectivity(0);
	mat6.setReflectivity(0);
	mat7.setReflectivity(0);
	mat8.setReflectivity(0);

	top.optimize();

	w = 1024;
	h = 600;

	/* run interactive mode */
	loop_sdl(rt, cam, update_callback, key_callback);

	/* render image */
	if (argc == 2 && !strcmp(argv[1], "-r"))
	{
		pyrit_verbosity = 2;
		rt.ambientOcclusion(300, 5.0f, 0.5f);
		DefaultSampler sampler(w, h);
		sampler.setOversample(2);
		sampler.setSubsample(1);
		rt.setSampler(&sampler);
		rt.render();
		sampler.getPixmap().writePNG("textures.png");
	}
	
	return 0;
}
