/*
 * vector.h: Vector3 class with Colour alias
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef VECTOR_H
#define VECTOR_H

#include <math.h>
#include <iostream>

#include "common.h"

using namespace std;

/**
 * three cell vector
 */
class Vector3
{
public:
	// data
	union {
		struct {
			Float x, y, z;
		};
		struct {
			Float r, g, b;
		};
		Float cell[3];
	};

	// constructors
	Vector3(): x(0.0f), y(0.0f), z(0.0f) {};
	Vector3(Float ax, Float ay, Float az): x(ax), y(ay), z(az) {};

	// index operator
	const Float &operator[](int index) const { return cell[index]; };

	bool operator==(Vector3 &v) const { return x==v.x && y==v.y && z==v.z; };

	// normalize
	Vector3 normalize()
	{
		Float f = 1.0f / mag();
		x *= f;
		y *= f;
		z *= f;
		return *this;
	};

	// get normalized copy
	friend Vector3 normalize(Vector3 &v)
	{
		const Float f = 1.0f / v.mag();
		return v * f;
	};

	// square magnitude, magnitude
	Float mag2() const	{ return x * x + y * y + z * z; };
	Float mag() const	{ return sqrtf(mag2()); };

	// negative
	Vector3 operator-() const { return Vector3(-x, -y, -z); };

	// accumulate
	Vector3 operator+=(const Vector3 &v)
	{
		x += v.x;
		y += v.y;
		z += v.z;
		return *this;
	};

	// cut
	Vector3 operator/=(const Float &f)
	{
		x /= f;
		y /= f;
		z /= f;
		return *this;
	};

	// sum
	friend Vector3 operator+(const Vector3 &a, const Vector3 &b)
	{
		return Vector3(a.x + b.x, a.y + b.y, a.z + b.z);
	};

	// difference
	friend Vector3 operator-(const Vector3 &a, const Vector3 &b)
	{
		return Vector3(a.x - b.x, a.y - b.y, a.z - b.z);
	};

	// dot product
	friend Float dot(const Vector3 &a, const Vector3 &b)
	{
		return a.x * b.x + a.y * b.y + a.z * b.z;
	};

	// cross product
	friend Vector3 cross(const Vector3 &a, const Vector3 &b)
	{
		return Vector3(a.y * b.z - a.z * b.y,
			a.z * b.x - a.x * b.z,
			a.x * b.y - a.y * b.x);
	};

	// product of vector and scalar
	friend Vector3 operator*(const Vector3 &v, const Float &f)
	{
		return Vector3(f * v.x, f * v.y, f * v.z);
	};

	friend Vector3 operator*(const Float &f, const Vector3 &v)
	{
		return v * f;
	};

	// scalar division
	friend Vector3 operator/(const Vector3 &v, const Float &f)
	{
		return Vector3(v.x / f, v.y / f, v.z / f);
	};

	friend Vector3 operator/(const Float &f, const Vector3 &v)
	{
		return Vector3(f / v.x, f / v.y, f / v.z);
	};

	// vector plus scalar
	friend Vector3 operator+(const Vector3 &v, const Float &f)
	{
		return Vector3(v.x + f, v.y + f, v.z + f);
	};

	// vector minus scalar
	friend Vector3 operator-(const Vector3 &v, const Float &f)
	{
		return Vector3(v.x - f, v.y - f, v.z - f);
	};

	// cell by cell product (only usable for colours)
	friend Vector3 operator*(const Vector3 &a,  const Vector3 &b)
	{
		return Vector3(a.x * b.x, a.y * b.y, a.z * b.z);
	};

	// write
	friend ostream & operator<<(ostream &st, const Vector3 &v)
	{
		return st << "(" << v.x << "," << v.y  << "," << v.z << ")";
	};

	// read
	friend istream & operator>>(istream &st, Vector3 &v)
	{
		char s[10];
		st.getline(s, 10, '(');
		st >> v.x;
		st.getline(s, 10, ',');
		st >> v.y;
		st.getline(s, 10, ',');
		st >> v.z;
		st.getline(s, 10, ')');
		return st;
	};
};

typedef Vector3 Colour;

class VectorPacket
{
public:
	union {
		__m128 ma[3];
		struct {
			__m128 mx;
			__m128 my;
			__m128 mz;
		};
		struct {
			float x[4];
			float y[4];
			float z[4];
		};
	};

	VectorPacket() {};
	VectorPacket(__m128 ax, __m128 ay, __m128 az):
		mx(ax), my(ay), mz(az) {};

	Vector3 getVector(int i) const
	{
		return Vector3(x[i], y[i], z[i]);
	};

	void normalize()
	{
		__m128 m,x,y,z;
		x = _mm_mul_ps(mx, mx); // x*x
		y = _mm_mul_ps(my, my); // y*y
		z = _mm_mul_ps(mz, mz); // z*z
		m = _mm_add_ps(x, y);
		m = _mm_add_ps(m, z);     // x*x + y*y + z*z
		m = _mm_sqrt_ps(m);
		m = _mm_div_ps(mOne, m);   // m = 1/sqrt(m)
		mx = _mm_mul_ps(mx, m);
		my = _mm_mul_ps(my, m);
		mz = _mm_mul_ps(mz, m);
	};

	friend VectorPacket operator+(const VectorPacket &a, const VectorPacket &b)
	{
		return VectorPacket(
			_mm_add_ps(a.mx, b.mx),
			_mm_add_ps(a.my, b.my),
			_mm_add_ps(a.mz, b.mz));
	};

	friend VectorPacket operator-(const VectorPacket &a, const VectorPacket &b)
	{
		return VectorPacket(
			_mm_sub_ps(a.mx, b.mx),
			_mm_sub_ps(a.my, b.my),
			_mm_sub_ps(a.mz, b.mz));
	};

	friend VectorPacket operator*(const VectorPacket &v,  const __m128 &m)
	{
		return VectorPacket(
			_mm_mul_ps(v.mx, m),
			_mm_mul_ps(v.my, m),
			_mm_mul_ps(v.mz, m));
	};

	friend VectorPacket operator/(const __m128 &m, const VectorPacket &v)
	{
		return VectorPacket(
			_mm_div_ps(m, v.mx),
			_mm_div_ps(m, v.my),
			_mm_div_ps(m, v.mz));
	};

	// write to character stream
	friend ostream & operator<<(ostream &st, const VectorPacket &v)
	{
		return st << "[" << v.getVector(0) << "," << v.getVector(1)
			<< "," << v.getVector(2) << "," << v.getVector(3) << ")";
	};

};

#endif
