/**
 * @file  kdtree.h
 * @brief KdTree class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef KDTREE_H
#define KDTREE_H

#include <iostream>
#include <fstream>
#include <assert.h>

#include "common.h"
#include "vector.h"
#include "scene.h"
#include "container.h"
#include "mempool.h"

using namespace std;

/**
 * a node of kd-tree
 */
class KdNode
{
	Float split;
	union {
		KdNode *children;
		ShapeList *shapes;
		int flags; /* 0,1,2 => x,y,z; 3 => leaf */
	};
public:
	KdNode() { shapes = new ShapeList(); assert((flags & 3) == 0); setLeaf(); };
	~KdNode();

	/** mark this node as leaf */
	void setLeaf() { flags |= 3; };
	bool isLeaf() const { return (flags & 3) == 3; };

	/** set split axis (this removes leaf flag) */
	void setAxis(int aAxis) { flags &= ~3; flags |= aAxis; };
	int getAxis() const { return flags & 3; };

	/** set split position (leaf) */
	void setSplit(Float aSplit) { split = aSplit; };
	const Float& getSplit() const { return split; };

	/** set children (non-leaf) */
	void setChildren(KdNode *node) { children = node; assert((flags & 3) == 0); };
	KdNode* getLeftChild() const { return (KdNode*)((size_t)children & ~3); };
	KdNode* getRightChild() const { return (KdNode*)(((size_t)children & ~3) + 16); };

	/** get shape list of the leaf node*/
	ShapeList* getShapes() const { return (ShapeList*)((size_t)shapes & ~3); };

	/** add shape to shape list */
	void addShape(const Shape* aShape) { getShapes()->push_back(aShape); };
};

/**
 * kd-tree
 */
class KdTree: public Container
{
	MemoryPool<KdNode> mempool;
	KdNode *root;
	const int max_depth;
	bool built;

	void recursive_build(KdNode *node, const BBox &bbox, int maxdepth);
	void recursive_load(istream &st, KdNode *node);
public:
	/** default constructor, maximum depth is set to 32 */
	KdTree(): Container(), mempool(64), root(NULL), max_depth(32), built(false) {};

	/** constructor which allows to se maximum tree depth (cannot be changed later) */
	KdTree(int maxdepth): Container(), mempool(64), root(NULL), max_depth(maxdepth), built(false) {};
	~KdTree() { if (root) delete root; };

	/** add shape pointer to the container */
	void addShape(const Shape* aShape) { Container::addShape(aShape); built = false; };
	const Shape *nearest_intersection(const Shape *origin_shape, const Ray &ray,
		Float &nearest_distance);
#ifndef NO_SIMD
	void packet_intersection(const Shape* const* origin_shapes, const RayPacket &rays,
		Float *nearest_distances, const Shape **nearest_shapes);
#endif
	void optimize() { build(); };

	/** build the tree (alias for 'optimize') */
	void build();
	bool isBuilt() const { return built; };
	KdNode *getRootNode() const { return root; };

	ostream & dump(ostream &st) const;
	istream & load(istream &st, Material *mat);
};

#endif
