/*
 * scene.h: classes for objects in scene
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SCENE_H
#define SCENE_H

#include <vector>
#include <typeinfo>

#include "common.h"
#include "sampler.h"
#include "vector.h"
#include "quaternion.h"

/**
 * ray
 */
class Ray
{
public:
	Vector3 o, dir;
	Ray(): o(), dir() {};
	Ray(const Vector3 &ao, const Vector3 &adir):
		o(ao), dir(adir) {};
};

/**
 * a camera
 */
class Camera
{
public:
	Vector3 eye, p, u, v;
	Float F;

	Camera(): eye(0,0,10), p(0,0,-1), u(-1,0,0), v(0,1,0), F(2.*tan(M_PI/8.)) {};
	Camera(const Vector3 &C, const Vector3 &ap, const Vector3 &au, const Vector3 &av):
		eye(C), p(ap), u(au), v(av), F(2.*tan(M_PI/8.)) {};
	Camera(const Vector3 &from, const Vector3 &lookat, const Vector3 &up):
		eye(from), F(2.*tan(M_PI/8.))
	{
		p = lookat - from; u = cross(up, p);
		p.normalize(); u.normalize();
		v = cross(p, u);
	};
	void setEye(const Vector3 &aeye) { eye = aeye; };
	void setAngle(const Float angle) { F = 2.*tan(angle/2.); };
	void rotate(const Quaternion &q);
	void move(const Float fw, const Float left, const Float up);

	Ray makeRay(Sample &samp)
	{
		Vector3 dir = p - (u*samp.x + v*samp.y)*F;
		dir.normalize();
		return Ray(eye, dir);
	};

	void makeRayPacket(Sample *samples, Ray *rays)
	{
		__m128 m1x,m1y,m1z;
		__m128 m2x,m2y,m2z;
		__m128 m;
		
		// m1(xyz) = u * samples[i].x
		m1x = _mm_set1_ps(u.x);
		m1y = _mm_set1_ps(u.y);
		m1z = _mm_set1_ps(u.z);
		m = _mm_set_ps(samples[0].x, samples[1].x, samples[2].x, samples[3].x);
		m1x = _mm_mul_ps(m1x, m);
		m1y = _mm_mul_ps(m1y, m);
		m1z = _mm_mul_ps(m1z, m);
		
		// m2(xyz) = v * samples[i].y
		m2x = _mm_set1_ps(v.x);
		m2y = _mm_set1_ps(v.y);
		m2z = _mm_set1_ps(v.z);
		m = _mm_set_ps(samples[0].y, samples[1].y, samples[2].y, samples[3].y);
		m2x = _mm_mul_ps(m2x, m);
		m2y = _mm_mul_ps(m2y, m);
		m2z = _mm_mul_ps(m2z, m);
		
		// m1(xyz) = (m1 + m2) = (u*samples[i].x + v*samples[i].y)
		m1x = _mm_add_ps(m1x, m2x);
		m1y = _mm_add_ps(m1y, m2y);
		m1z = _mm_add_ps(m1z, m2z);
		
		// m1(xyz) = m1*F = (u*samples[i].x + v*samples[i].y)*F
		m = _mm_set_ps(F,F,F,F);
		m1x = _mm_mul_ps(m1x, m);
		m1y = _mm_mul_ps(m1y, m);
		m1z = _mm_mul_ps(m1z, m);
		
		// m1(xyz) = p - m1 = p - (u*samples[i].x + v*samples[i].y)*F = dir
		m2x = _mm_set1_ps(p.x);
		m2y = _mm_set1_ps(p.y);
		m2z = _mm_set1_ps(p.z);
		m2x = _mm_sub_ps(m2x, m1x);
		m2y = _mm_sub_ps(m2y, m1y);
		m2z = _mm_sub_ps(m2z, m1z);
		
		// normalize dir
		m1x = _mm_mul_ps(m2x, m2x); // x*x
		m1y = _mm_mul_ps(m2y, m2y); // y*y
		m1z = _mm_mul_ps(m2z, m2z); // z*z
		m = _mm_add_ps(m1x, m1y);   // x*x + y*y
		m = _mm_add_ps(m, m1z);     // m = x*x + y*y + z*z
		m = _mm_sqrt_ps(m);         // m = sqrt(m)
		m2x = _mm_div_ps(m2x, m);   // dir(xyz) /= m
		m2y = _mm_div_ps(m2y, m);
		m2z = _mm_div_ps(m2z, m);
		
		for (int i = 0; i < 4; i++)
		{
			Vector3 dir(((float*)&m2x)[3-i], ((float*)&m2y)[3-i], ((float*)&m2z)[3-i]);
			rays[i] = Ray(eye, dir);
		}
	};
};

/**
 * light object
 */
class Light
{
public:
	Vector3 pos;
	Colour colour;
	bool cast_shadows;

	Light():
		pos(Vector3(0,0,0)), colour(Colour(1,1,1)), cast_shadows(true) {};
	Light(const Vector3 &position, const Colour &acolour):
		pos(position), colour(acolour), cast_shadows(true) {};
	void castShadows(bool cast) { cast_shadows = cast; };
};

/**
 * axis-aligned bounding box
 */
class BBox
{
public:
	Vector3 L;
	Vector3 H;
	BBox(): L(), H() {};
	BBox(const Vector3 aL, const Vector3 aH): L(aL), H(aH) {};
	Float w() { return H.x-L.x; };
	Float h() { return H.y-L.y; };
	Float d() { return H.z-L.z; };
	bool intersect(const Ray &ray, Float &a, Float &b);
};

#endif
