/*
 * pixmap.cc: 2D image class
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include <stdio.h>

#ifdef HAVE_PNG
#	include <png.h>
#endif

#include "pixmap.h"

unsigned char *Pixmap::getCharData() const
{
	unsigned char *cdata = new unsigned char[w*h*3];
	Float *fd = fdata;

#ifdef NO_SSE
	for (unsigned char *cd = cdata; cd != cdata + w*h*3; cd++, fd++)
		if (*fd > 1.0)
			*cd = 255;
		else
			*cd = (unsigned char)(*fd * 255.0);
#else
	__m128 cmax = _mm_set_ps1(255.0);
	__m64 m;
	for (unsigned char *cd = cdata; cd < cdata + w*h*3; cd += 4, fd += 4)
	{
		m = _mm_cvtps_pi16(_mm_mul_ps(cmax,
			_mm_min_ps(mOne, _mm_set_ps(fd[3],fd[2],fd[1],fd[0]))));
		for (int i = 0; i < 4; i++)
			cd[i] = ((unsigned char *)&m)[i<<1];
	}
#endif

	return cdata;
}

int Pixmap::writePNG(const char *fname) const
{
#ifndef HAVE_PNG
	return -3;
#else
	int y;
	FILE *f;
	png_structp png;   /* PNG data */
	png_infop pnginfo; /* PNG info */
	unsigned char *cdata, *d;

	if ((f = fopen(fname, "wb")) == NULL)
		return -1;

	png = png_create_write_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0);
	if (!png) {
		fclose(f);
		return -2;
	}

	pnginfo = png_create_info_struct(png);
	if (!pnginfo) {
		fclose(f);
		png_destroy_write_struct(&png, 0);
		return -2;
	}

	if (setjmp(png_jmpbuf(png))) {
		fclose(f);
		png_destroy_info_struct(png, &pnginfo);
		png_destroy_write_struct(&png, &pnginfo);
		return -2;
	}

	/* predat knihovne PNG ukazatel na soubor */
	png_init_io(png, f);

	/* parametry PNG */
	png_set_compression_level(png, Z_BEST_COMPRESSION);
	png_set_IHDR(png, pnginfo, w, h, 8, PNG_COLOR_TYPE_RGB,
		PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT,
		PNG_FILTER_TYPE_DEFAULT);
	png_set_sRGB(png, pnginfo, PNG_sRGB_INTENT_PERCEPTUAL);
	png_set_sRGB_gAMA_and_cHRM(png, pnginfo, PNG_INFO_sRGB);

	/* zapsat hlavicku */
	png_write_info(png, pnginfo);

	/* zapsat data */
	d = cdata = getCharData();
	for (y = 0; y < h; y++, d += w * 3)
		png_write_row(png, (png_byte *) d);

	/* ukoncit soubor a uvolnit pomocne struktury */
	png_write_end(png, pnginfo);
	png_destroy_info_struct(png, &pnginfo);
	png_destroy_write_struct(&png, 0);

	fclose(f);

	delete[] cdata;

	return 0;
#endif
}
