/**
 * @file  raytracermodule.h
 * @brief Ray tracer module for Python
 *
 * This file is part of Pyrit Ray Tracer.
 *
 * Copyright 2006, 2007, 2008  Radek Brich
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef RAYTRACERMODULE_H
#define RAYTRACERMODULE_H

#include <Python.h>
#include "raytracer.h"
#include "octree.h"
#include "kdtree.h"


#define TYPE_OBJECT(name, basicsize, dealloc, flags, doc, methods, members, base, init) \
{ \
	PyObject_HEAD_INIT(NULL) \
	0,                          /* ob_size */ \
	(name),                     /* tp_name */ \
	(basicsize),                /* tp_basicsize*/ \
	0,                          /* tp_itemsize*/ \
	(dealloc),                  /* tp_dealloc*/ \
	0,                          /* tp_print*/\
	0,                          /* tp_getattr*/\
	0,                          /* tp_setattr*/\
	0,                          /* tp_compare*/\
	0,                          /* tp_repr*/\
	0,                          /* tp_as_number*/\
	0,                          /* tp_as_sequence*/\
	0,                          /* tp_as_mapping*/\
	0,                          /* tp_hash */\
	0,                          /* tp_call*/\
	0,                          /* tp_str*/\
	PyObject_GenericGetAttr,    /* tp_getattro*/\
	0,                          /* tp_setattro*/\
	0,                          /* tp_as_buffer*/\
	(flags),                    /* tp_flags*/\
	(doc),                      /* tp_doc */\
	0,                          /* tp_traverse */\
	0,                          /* tp_clear */\
	0,                          /* tp_richcompare */\
	0,                          /* tp_weaklistoffset */\
	0,                          /* tp_iter */\
	0,                          /* tp_iternext */\
	(methods),                  /* tp_methods */\
	(members),                  /* tp_members */\
	0,                          /* tp_getset */\
	(base),                     /* tp_base */\
	0,                          /* tp_dict */ \
	0,                          /* tp_descr_get */ \
	0,                          /* tp_descr_set */ \
	0,                          /* tp_dictoffset */ \
	(init),                     /* tp_init */ \
	0,                          /* tp_alloc */ \
	0,                          /* tp_new */ \
	0,                          /* tp_free */ \
}

/** Light object */
typedef struct {
	PyObject_HEAD
	Light *light;
} LightObject;

/** Camera object */
typedef struct {
	PyObject_HEAD
	Camera *camera;
} CameraObject;

/** Pixmap object */
typedef struct {
	PyObject_HEAD
	const Pixmap *pixmap;
} PixmapObject;

/** abstract TextureMap object */
typedef struct {
	PyObject_HEAD
	TextureMap *texturemap;
} TextureMapObject;

/** PlanarMap object - inherits TextureMap */
typedef struct {
	TextureMapObject texturemap;
} PlanarMapObject;

/** CubicMap object - inherits TextureMap */
typedef struct {
	TextureMapObject texturemap;
} CubicMapObject;

/** CylinderMap object - inherits TextureMap */
typedef struct {
	TextureMapObject texturemap;
} CylinderMapObject;

/** SphereMap object - inherits TextureMap */
typedef struct {
	TextureMapObject texturemap;
} SphereMapObject;

/** abstract ColourMap object */
typedef struct {
	PyObject_HEAD
	ColourMap *colourmap;
} ColourMapObject;

/** LinearColourMap object - inherits ColourMap */
typedef struct {
	ColourMapObject colourmap;
} LinearColourMapObject;

/** BoundColourMap object - inherits ColourMap */
typedef struct {
	ColourMapObject colourmap;
	Float *bounds;
	Colour *colours;
} BoundColourMapObject;

/** abstract Texture object */
typedef struct {
	PyObject_HEAD
	Texture *texture;
} TextureObject;

/** ImageTexture object - inherits Texture */
typedef struct {
	TextureObject texture;
} ImageTextureObject;

/** CheckersTexture object - inherits Texture */
typedef struct {
	TextureObject texture;
} CheckersTextureObject;

/** CloudTexture object - inherits Texture */
typedef struct {
	TextureObject texture;
} CloudTextureObject;

/** Material object */
typedef struct {
	PyObject_HEAD
	Material *material;
} MaterialObject;

/** Vertex object */
typedef struct {
	PyObject_HEAD
	Vertex *vertex;
} VertexObject;

/** NormalVertex object - inherits Vertex*/
typedef struct {
	VertexObject vertex;
} NormalVertexObject;

/** abstract Shape object */
typedef struct {
	PyObject_HEAD
	Shape *shape;
} ShapeObject;

/** Triangle object - inherits Shape */
typedef struct {
	ShapeObject shape;
} TriangleObject;

/** Sphere object - inherits Shape */
typedef struct {
	ShapeObject shape;
} SphereObject;

/** Box object - inherits Shape */
typedef struct {
	ShapeObject shape;
} BoxObject;

/** abstract Sampler object */
typedef struct {
	PyObject_HEAD
	Sampler *sampler;
} SamplerObject;

/** DefaultSampler object - inherits Sampler */
typedef struct {
	SamplerObject sampler;
} DefaultSamplerObject;

/** Container object */
typedef struct {
	PyObject_HEAD
	Container *container;
} ContainerObject;

/** Octree object - inherits Container */
typedef struct {
	ContainerObject container;
} OctreeObject;

/** KdTree object - inherits Container */
typedef struct {
	ContainerObject container;
} KdTreeObject;

/** main Raytracer object */
typedef struct {
	PyObject_HEAD
	Raytracer *raytracer;
	vector<PyObject*> *children;
} RaytracerObject;

#endif
