#!/usr/bin/env python3

import sys
sys.path.insert(0, '..')

from tuikit.tableview import *
import unittest


class TestTableView(unittest.TestCase):
    def setUp(self):
        self.abs_model = AbstractTableModel()
        self.model = TableModel()

    def test_initial(self):
        self._test_empty_model(self.abs_model)
        self._test_empty_model(self.model)

    def test_one(self):
        self._insert_data()
        self._fill_data()
        self._insert_headers()
        self._test_empty_headers()
        self._fill_headers()
        self._remove_data()
        self._clear_data()

    def test_two(self):
        self._insert_headers()
        self._insert_data()
        self._test_empty_headers()
        self._fill_headers()
        self._fill_data()

    def _test_empty_model(self, model):
        self.assertEqual(model.row_count(), 0)
        self.assertEqual(model.column_count(), 0)
        self.assertRaises(IndexError, model.get_row, 0)
        self.assertRaises(IndexError, model.get_column, 0)
        self.assertRaises(IndexError, model.get_item, 0, 0)

    def _insert_data(self):
        for _i in range(3):
            self.model.insert_row(0)
            self.model.insert_column(0)
        self.assertEqual(self.model.row_count(), 3)
        self.assertEqual(self.model.column_count(), 3)

    def _fill_data(self):
        for row in range(3):
            for col in range(3):
                self.model.set_item(row, col, row*3 + col+1)
        self.assertEqual(self.model.get_row(0), [1,2,3])
        self.assertEqual(self.model.get_row(1), [4,5,6])
        self.assertEqual(self.model.get_row(2), [7,8,9])
        self.assertEqual(self.model.get_column(0), [1,4,7])

    def _insert_headers(self):
        self.model.set_num_headers(2, 2)
        self.assertEqual(self.model.row_header_count(), 2)
        self.assertEqual(self.model.column_header_count(), 2)

    def _test_empty_headers(self):
        self.assertEqual(self.model.get_row_header(0, 0), None)
        self.assertEqual(self.model.get_row_header(2, 1), None)
        self.assertEqual(self.model.get_column_header(0, 0), None)
        self.assertEqual(self.model.get_column_header(2, 1), None)

    def _fill_headers(self):
        for i in range(3):
            for header in range(2):
                self.model.set_column_header(i, header, header*3 + i + 1)
                self.model.set_row_header(i, header, header*3 + i + 1)
        self.assertEqual(self.model.get_column_header(1, 0), 2)
        self.assertEqual(self.model.get_column_header(1, 1), 5)
        self.assertEqual(self.model.get_row_header(1, 0), 2)
        self.assertEqual(self.model.get_row_header(1, 1), 5)

    def _remove_data(self):
        self.model.remove_row(1)
        self.assertEqual(self.model.get_row(0), [1,2,3])
        self.assertEqual(self.model.get_row(1), [7,8,9])
        self.assertEqual(self.model.get_column(0), [1,7])
        self.model.remove_column(1)
        self.assertEqual(self.model.get_row(0), [1,3])
        self.assertEqual(self.model.get_row(1), [7,9])
        self.assertEqual(self.model.get_column(1), [3,9])
        self.assertRaises(IndexError, self.model.get_row, 2)
        self.assertRaises(IndexError, self.model.get_column, 2)

    def _clear_data(self):
        self.model.clear()


if __name__ == '__main__':
    unittest.main()

