#include "SDL.h"
#include "SDL_ttf.h"

#include <map>
#include <vector>


class GlyphRenderer
{
public:
	GlyphRenderer();
	~GlyphRenderer();

	void open_font(const char *fname_regular, const char *fname_bold, int ptsize);
	void close_font();

	SDL_Surface *render_glyph(Uint16 ch);

	int get_cell_width() { return _cell_width; };
	int get_cell_height() { return _cell_height; };

private:
	TTF_Font *_font_regular;
	TTF_Font *_font_bold;
	int _cell_width;
	int _cell_height;
	std::map<Uint16, SDL_Surface*> _cache;
};


struct TerminalCell
{
	Uint16 ch;
	Uint16 attr;
};


class TerminalScreen
{
public:
	TerminalScreen(): _screen_surface(NULL), _cells(NULL), _render() {};
	~TerminalScreen();

	void select_font(const char *fname_regular, const char *fname_bold, int ptsize);
	void resize(int pxwidth, int pxheight);

	void erase();
	void putch(int x, int y, Uint16 ch, Uint16 attr);
	void commit();

private:
	SDL_Surface *_screen_surface;
	TerminalCell *_cells;
	GlyphRenderer _render;

	int _pixel_width;  // terminal window width in pixels
	int _pixel_height;
	int _width;  // width in characters
	int _height; // height in characters
	int _cell_width;  // character cell width in pixels
	int _cell_height;

	void _reset_cells();
};


enum class EventType : Uint8
{
	quit,
	resize,
	keypress,
	mousedown,
	mouseup,
	mousemove,
	mousewheel
};


struct WindowEvent
{
	EventType type;
};


struct KeyboardEvent
{
	EventType type;
};


struct MouseEvent
{
	EventType type;
};


union Event
{
	EventType type;
	WindowEvent window;
	KeyboardEvent key;
	MouseEvent mouse;
};


class Terminal
{
public:
	Terminal();
	~Terminal();

	void select_font(const char *fname_regular, const char *fname_bold, int ptsize)
		{ _screen.select_font(fname_regular, fname_bold, ptsize); };
	void resize(int pxwidth, int pxheight) { _screen.resize(pxwidth, pxheight); };

	void erase() { _screen.erase(); };
	void putch(int x, int y, Uint16 ch) { _screen.putch(x, y, ch, _attr); };
	void commit() { _screen.commit(); };

	Uint16 prepare_attr(Uint8 fg, Uint8 bg, Uint8 style) { return fg | bg << 8 | style << 16; };
	void set_attr(Uint16 value) { _attr = value; };

	void set_cursor(int x, int y) { _cursor_x = x; _cursor_y = y; };
	void show_cursor(bool visible) { _cursor_visible = visible; };

	void get_next_event(Event &event);

private:
	TerminalScreen _screen;
	Uint16 _attr;
	int _cursor_x;
	int _cursor_y;
	bool _cursor_visible;
};

