from tuikit.core.widget import Widget
from tuikit.core.signal import Signal


class Button(Widget):

    """Clickable button."""

    def __init__(self, label='btn'):
        """Create button with given label, size according to label."""
        Widget.__init__(self)
        self.allow_focus = True

        #: Button label.
        self._label = ''
        #: Text or graphics to be added before label
        self.prefix = '['
        #: Text or graphics to be added after label
        self.suffix = ']'
        #: How should label be aligned if button has excess space - center | left | right
        self.align = 'center'
        #: Padding between prefix/suffix and label
        self.padding = 1

        self.color = 'default'
        self.color_active = 'default on red'
        self.highlight = False

        self.sig_clicked = Signal()

        self.label = label

    @property
    def label(self):
        """Button label."""
        return self._label

    @label.setter
    def label(self, value):
        self._label = value
        w = len(value) + len(self.prefix) + len(self.suffix) + 2 * self.padding
        self.sizereq.update(w, 1)

    def set_theme(self, theme):
        Widget.set_theme(self, theme)
        self.color = theme.button
        self.color_active = theme.button_active

    def _get_color(self):
        if self.has_focus():
            return self.color_active
        return self.color

    def draw(self, buffer):
        Widget.draw(self, buffer)
        pad = self.width - len(self.label) - len(self.prefix) - len(self.suffix)
        lpad, rpad = self._divide_padding(pad)
        with buffer.attr(self._get_color()):
            # prefix
            buffer.puts(self.prefix)
            pos = len(self.prefix)
            # left pad
            buffer.puts(' ' * lpad, pos)
            pos += lpad
            # label
            buffer.puts(self.label, pos)
            pos += len(self.label)
            # right pad
            buffer.puts(' ' * rpad, pos)
            pos += rpad
            # suffix
            buffer.puts(self.suffix, pos)

    def on_mousedown(self, ev):
        self.highlight = True
        self.redraw()

    def on_mouseup(self, ev):
        self.highlight = False
        self.redraw()

        if self.enclose(ev.px, ev.py):
            self.emit('click')

    def on_keypress(self, ev):
        if ev.keyname == 'enter':
            self.emit('click')

    def _divide_padding(self, pad):
        # default is 'left'
        lpad, rpad = 0, pad
        if self.align == 'center':
            lpad = pad // 2
            rpad = pad - lpad
        elif self.align == 'right':
            lpad, rpad = pad, 0
        return lpad, rpad

