# -*- coding: utf-8 -*-

from tuikit.container import Container
from tuikit.button import Button


class Window(Container):
    
    '''Window widget.
    
    It represents part of screen with border, close button and contents.
    Window can be moved, resized or closed by user.'''
    
    def __init__(self, width=40, height=10):
        '''Create window of requested size.'''
        Container.__init__(self, width, height)

        self.connect('draw', self.on_draw)
        self.connect('mousedown', self.on_mousedown)
        self.connect('mouseup', self.on_mouseup)
        self.connect('mousemove', self.on_mousemove)
        self.connect('resize', self.on_resize)

        #: Window title.
        self.title = ''
        
        self._closebutton = True
        
        #: Allow user to resize window.
        self.resizable = True
        
        #: Allow user to move window.
        self.movable = True

        self.resizing = False
        self.moving = False

        self.closebtn = Button('x')
        self.closebtn.allowlayout = False
        self.closebtn.x = self.width - 5
        self.closebtn.width = 3
        self.closebtn.connect('click', self.on_closebtn)
        self.closebtn.bg = 'controls'
        self.closebtn.bghi = 'controls-active'
        self.add(self.closebtn)

        self.borders = (1, 1, 1, 1)
        self.colorprefix = 'window:'


    @property
    def closebutton(self):
        '''Show/hide close button.'''
        return self._closebutton

    @closebutton.setter
    def closebutton(self, value):
        self._closebutton = value
        self.closebtn.hidden = not value


    def on_draw(self, screen, x, y):
        screen.pushcolor('normal')
        screen.frame(x, y, self.width, self.height)

        if self.resizable:
            if self.resizing:
                screen.pushcolor('controls-active')
            else:
                screen.pushcolor('controls')
            screen.puts(x + self.width - 2, y + self.height - 1, '─┘') # '━┛'
            screen.popcolor()

        if self.title:
            screen.puts(x + (self.width - len(self.title))//2, y, self.title)

        screen.fill(x+1, y+1, self.width-2, self.height-2)
        screen.popcolor()


    def on_mousedown(self, ev):
        self.dragstart = (ev.wx, ev.wy)
        if self.resizable and ev.wx >= self.width - 1 and ev.wy >= self.height - 1:
            self.resizing = True
        elif self.movable:
            self.moving = True
        self.origsize = (self.width, self.height)

        self.redraw(True)


    def on_mouseup(self, ev):
        if self.resizing:
            self.width = self.origsize[0] + ev.wx - self.dragstart[0]
            self.height = self.origsize[1] + ev.wy - self.dragstart[1]
            self.resizing = False
            self.emit('resize')
        elif self.moving:
            self.x = ev.px - self.dragstart[0]
            self.y = ev.py - self.dragstart[1]
            self.moving = False

        self.redraw(True)


    def on_mousemove(self, ev):
        if ev.px == self.x + self.dragstart[0] \
        and ev.py == self.y + self.dragstart[1]:
            return

        #if x > self.parent.width-self.width:
        #    return

        if self.resizing:
            self.width = self.origsize[0] + ev.wx - self.dragstart[0]
            self.height = self.origsize[1] + ev.wy - self.dragstart[1]
            self.emit('resize')
        elif self.moving:
            self.x = ev.px - self.dragstart[0]
            self.y = ev.py - self.dragstart[1]

        self.redraw(True)


    def on_resize(self):
        self.closebtn.x = self.width - 5


    def on_closebtn(self):
        self.hide()


