# -*- coding: utf-8 -*-
'''PyGame driver.'''

import pygame
import logging

from tuikit.driver import Driver
from tuikit.common import Coords, Size


class CharCache:
    
    '''CharCache should implement character cache, as the name suggests.
    
    Currently it does not, characters ale rendered directly from TTF.
    
    '''
    
    def __init__(self):
        self.font = pygame.font.SysFont('dejavusansmono,liberationmono,freemono', 14)
        # get advance of some random char (all should be same in monospace font)
        advance = self.font.metrics('Q')[0][4]
        height = self.font.get_height()
        self.charsize = Size(advance, height)
        self.ascent = self.font.get_ascent()
        
        # choose self.render() implementation
        if hasattr(self.font, 'render_glyph'):
            self.render = self.render_glyph
        else:
            self.render = self.render_noglyph

    def render_glyph(self, screen, x, y, c, fgcolor, bgcolor):
        '''Render using render_glyph and metrics.
        
        This is the correct way, but the output seems same as of render_noglyph
        and this implementation requires patching PyGame to work.
        
        This implements render() method. See render_noglyph for other implementation.
        
        '''
        # render character, get metrics
        surface = self.font.render_glyph(c, True, fgcolor, bgcolor)
        metrics = self.font.metrics(c)[0]
        minx, maxx, miny, maxy, advance = metrics
        height, ascent = self.charsize.h, self.ascent
        
        # clips origin and area of rendered character according to metrics
        startx, starty = 0, 0
        if minx < 0:
            startx = abs(minx)
            minx = 0
        if maxy > ascent:
            starty = maxy - ascent
            maxy -= starty
        if ascent - miny > height:
            miny = ascent - height
        if maxx > advance:
            maxx = advance
        
        # draw background
        dest = Coords(x * self.charsize.w, y * self.charsize.h)
        screen.fill(bgcolor, pygame.Rect(dest.x, dest.y, self.charsize.w, self.charsize.h))
        
        # draw character
        dest.x += minx
        dest.y += ascent - maxy
        area = pygame.Rect(startx, starty, maxx - minx, maxy - miny)
        screen.blit(surface, tuple(dest), area)
    
    def render_noglyph(self, screen, x, y, c, fgcolor, bgcolor):
        '''Render character using normal text rendering.
        
        This implements render() method. See render_glyph for other implementation.
        
        '''       
        # render character, get metrics
        surface = self.font.render(c, True, fgcolor, bgcolor)
        metrics = self.font.metrics(c)[0]
        minx = metrics[0]
        startx = 0
        if minx < 0:
            startx = abs(minx)
        
        # draw background
        dest = Coords(x * self.charsize.w, y * self.charsize.h)
        screen.fill(bgcolor, pygame.Rect(dest.x, dest.y, self.charsize.w, self.charsize.h))
        
        # draw character
        area = pygame.Rect(startx, 0, self.charsize.w, self.charsize.h)
        screen.blit(surface, tuple(dest), area)


class DriverPygame(Driver):
    
    '''PyGame driver class.'''
    
    key_map = {
        pygame.K_ESCAPE     : 'escape',
        pygame.K_TAB        : 'tab',
        pygame.K_RETURN     : 'enter',
        pygame.K_BACKSPACE  : 'backspace',
        pygame.K_F1         : 'f1',
        pygame.K_F2         : 'f2',
        pygame.K_F3         : 'f3',
        pygame.K_F4         : 'f4',
        pygame.K_F5         : 'f5',
        pygame.K_F6         : 'f6',
        pygame.K_F7         : 'f7',
        pygame.K_F8         : 'f8',
        pygame.K_F9         : 'f9',
        pygame.K_F10        : 'f10',
        pygame.K_F11        : 'f11',
        pygame.K_F12        : 'f12',
        pygame.K_INSERT     : 'insert',
        pygame.K_DELETE     : 'delete',
        pygame.K_HOME       : 'home',
        pygame.K_END        : 'end',
        pygame.K_PAGEUP     : 'pageup',
        pygame.K_PAGEDOWN   : 'pagedown',
        pygame.K_UP         : 'up',
        pygame.K_DOWN       : 'down',
        pygame.K_LEFT       : 'left',
        pygame.K_RIGHT      : 'right',
        pygame.K_PRINT      : 'print',
        pygame.K_SCROLLOCK  : 'scrollock',
        pygame.K_PAUSE      : 'pause',
        }
    
    color_map = {
        'black'   : (0,0,0),
        'blue'    : (0,0,0),
        'cyan'    : (0,0,0),
        'green'   : (0,0,0),
        'magenta' : (0,0,0),
        'red'     : (0,0,0),
        'white'   : (0,0,0),
        'yellow'  : (0,0,0),
        }
    
    def __init__(self):
        '''Initialize instance attributes'''
        Driver.__init__(self)
        self.log = logging.getLogger('tuikit')
        self.screen = None
        self.size.w, self.size.h = 120, 40  # screen size in characters
        self.charcache = None
        self.charsize = Size(16, 8)  # character size in pixels
        
        self.colorprefix = [] # stack of color prefixes

    def start(self, mainfunc):
        pygame.init()
        self.charcache = CharCache()
        self.charsize = self.charcache.charsize
        mode = self.size.w * self.charsize.w, self.size.h * self.charsize.h
        self.screen = pygame.display.set_mode(mode)
        mainfunc()


    ## input ##
    
    def getevents(self, timeout=None):
        '''Process input, return list of events.'''
        events = []
        for ev in pygame.event.get():
            if ev.type == pygame.MOUSEMOTION:
                pass
            elif ev.type == pygame.MOUSEBUTTONUP:
                pass
            elif ev.type == pygame.MOUSEBUTTONDOWN:
                pass
            elif ev.type == pygame.KEYDOWN:
                if ev.key in self.key_map:
                    events.append(('keypress', self.key_map[ev.key], None))
                elif ev.unicode:
                    events.append(('keypress', None, ev.unicode))
            elif ev.type == pygame.VIDEORESIZE:
                #self.size.h, self.size.w = self.screen.getmaxyx()
                events.append(('resize',))
            elif ev.type == pygame.QUIT:
                events.append(('quit',))
            else:
                self.log.warning('Unknown PyGame event: %r', ev.type)
        return events


    ## drawing ##
    
    def erase(self):
        '''Clear screen.'''
        self.screen.fill(self.color_map['black'])
    
    def putch(self, x, y, c):
        if not self.clipstack.test(x, y):
            return
        fgcolor = (255,255,255)
        bgcolor = (70,70,70)
        self.charcache.render(self.screen, x, y, c, fgcolor, bgcolor)

    def commit(self):
        '''Commit changes to the screen.'''
        pygame.display.flip()


    ## colors ##
    
    def setcolor(self, name, desc):
        '''Define color name.
        
        name - name of color (e.g. 'normal', 'active')
        desc - color description - foreground, background, attributes (e.g. 'black on white, bold')
        
        '''

    def pushcolor(self, name):
        '''Add color on top of stack and use this color for following output.'''
    
    def popcolor(self):
        '''Remove color from top of stack and use new top color for following output.'''

    def pushcolorprefix(self, name):
        self.colorprefix.append(name)

    def popcolorprefix(self):
        self.colorprefix.pop()


    ## cursor ##

    def showcursor(self, x, y):
        '''Set cursor to be shown at x, y coordinates.'''

    def hidecursor(self):
        '''Hide cursor.'''

