# -*- coding: utf-8 -*-

class EventSource:
    
    '''Event source base class.'''

    def __init__(self):
        self.event = dict()

    def addevents(self, *events):
        '''Create new events with empty handler list.'''
        for event in events:
            self.event[event] = []

    def connect(self, event, handler):
        '''Add handler to handler list of the event.'''
        if event in list(self.event.keys()):
            self.event[event].append(handler)
        else:
            raise KeyError('Event %s not known.', event)

    def disconnect(self, event, handler=None):
        '''Remove handler from event's handler list.

        If no handler is given, remove all handlers.

        '''
        if event in list(self.event.keys()):
            if handler:
                self.event[event].remove(handler)
            else:
                self.event[event] = []
        else:
            raise KeyError('Event %s not known.', event)

    def handle(self, event, *args, **kwargs):
        '''Call all handlers from event's handler list.

        This is used when user defined handlers are to be called.

        '''
        handled = False
        for handler in self.event[event]:
            res = handler(*args, **kwargs)
            if res:
                handled = True
        return handled

    def emit(self, event, *args, **kwargs):
        '''Emit event.

        This is used by original event source when the event is detected.

        '''
        try:
            getattr(self, event)(*args, **kwargs)
        except AttributeError:
            self.handle(event, *args, **kwargs)

